/****************************************************
 Include file for the BancorMS.dll, core library
 of the K2s device.

				Copyright bancor
 ****************************************************/
#ifndef K2SLIB
#define K2SLIB

#if defined (__cplusplus)
extern "C"	{
#endif

/**
* Returns currently attached device composition.
* @return Bitewise result, defining device composition. See K2sComp values.
* @see K2sComp
*/
unsigned long BMSCompositionL (void);

/**
* Returns the number of the virtual COM0 port, if any. 0 means absent.
*/
unsigned short BMSCOM0 (BMSHandle hDevice);

/**
* Returns the number of the virtual COM0 port, if any. 0 means absent.
*/
unsigned short BMSCOM1 (BMSHandle hDevice);

/** Initialization function. Returned BMSHandle to be released with BMSClose.
* @param <b>ConfigPath</b>	 Configuration folder path.<BR>
*									In this folder the sw will look for calibration files
*									and OCR databases.<BR>
*									If NULL, default init path is used.
* @param <b>SerialNumber</b> Serial number of the K2s to open (i.e. "216000").
*								If NULL, the first K2s is opened, if any.
* @return INVALID_HANDLE_VALUE or valid handle.
* @see BMSClose
*/
BMSHandle BMSInitializeSN (const char* ConfigPath, const char* SerialNumber);

/** Releases BMSHandle opended with initalization functios.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	BMSResult
* @see		BMSInitializeSN
*/
BMSResult BMSClose	(BMSHandle hDevice);

/** SW reset. Not supported by some HW versions.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	BMSResult 
*/
BMSResult BMSReset	(BMSHandle hDevice);

/** Current device status.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	Bitwise unsigned long, defining current device status. See K2sStatus.
* @see		K2sStatus
*/
unsigned long BMSReadStatusL (BMSHandle hDevice);

/** Returns in "version" a string describing SW version of BancorMS.dll and FW version.<BR>
* For example "SW Version 2.1.0.9, FW Version 0x2207".
* @param	hDevice		(in)		BMSHandle returned by initialization function.	
* @param	version		(in out)	Buffer allocated by the calling application. If version is null or size is 0, no data is copied and the required size (including NULL terminator) is returned.
* @param	size		(in)            Size of the buffer allocated by the calling application. If 0, no data is copied and the required size (including NULL terminator) is returned.
* @return	short                           Required or copied size. -1 in case of error. 
*/
short BMSVersion (BMSHandle hDevice, char *version, short size);

/** Returns in "SerialNumber" a string with the serial number of the device currently attached.
* Empty string if no device is detected or if device does not support this function.
* @param	hDevice		(in)		BMSHandle returned by initialization function.
* @param	SerialNumber	(in out)        Buffer allocated by the calling application. If version is null or size is 0, no data is copied and the required size (including NULL terminator) is returned.
* @param	size		(in)            Size of the buffer allocated by the calling application. If 0, no data is copied and the required size (including NULL terminator) is returned.
* @return	short                           Required or copied size. -1 in case of error. 
*/
short BMSReadSN(BMSHandle hDevice, char *SerialNumber, short size);

/** Returns Vendor Id, Product Id and firmware version of the connected device.
* Empty strings if no device is detected.
* @param	hDevice			BMSHandle returned by initialization function.	
* @param	vid			(in out)	Vendor Id (see USB specification). If pointer is NULL, this field is skipped. Hexadecimal format.
* @param	pid			(in out)	As vid, but returning Product Id.
* @param	fw			(in out)	As vid, but returning firmware version.
* @return	BMSResult 
*/
BMSResult BMSDeviceInfo (BMSHandle hDevice, unsigned short *vid, unsigned short *pid, unsigned short *fw);

/** Gets last result.
* @return	BMSResult 
*/
BMSResult BMSGetLastResult(void);

/** Gets description of a given error.
* @param    ErrN        (in)            Error code
* @param	ErrorString	(in out)        Buffer allocated by the calling application. If version is null or size is 0, no data is copied and the required size (including NULL terminator) is returned.
* @param	size		(in)            Size of the buffer allocated by the calling application. If 0, no data is copied and the required size (including NULL terminator) is returned.
* @return	short 
*/
short BMSErrMsg (BMSResult ErrN, char *ErrorString, short size);

/** Gets last result description.
* @return	BMSResult 
*/
BMSResult BMSErrorString (char *ErrorString, short size);	

/** 
* Internal use only 
*/
void BMSSetLastResult (BMSResult result);

/** Stopping current acquisition/operation.<BR>
* After BMSStop( , 1) has produced the desired effect,
* it must be called BMSStop( , 0) before performing a new acquisition/operation.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	value		0 to stop. 1 to re-enable.
*/
BMSResult BMSStop (BMSHandle hDevice, short value); // value > 0 - stop ; value <= 0 re-enable (needed after a stop)

/** Performs white and black calibration of the different acquisition modes, 
* using a white sheet of paper.<BR>
* Calibration is mandatory and it is required for each K2s that is connected to the PC.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	BMSResult
* @see BMSCheckCalibration.
*/
BMSResult BMSCalibrate			(BMSHandle hDevice);

/** Checks if the connected device is calibrated.
* @param	hDevice		BMSHandle returned by initialization function.
* @return	BMSResult
* @see BMSCalibrate
*/
BMSResult BMSCheckCalibration	(BMSHandle hDevice);


/** Changes acquisition brightness.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	BrightF		Between -100 (darkest) to +100.	Front side.
* @param	BrightR		As BrightF for Rear side.
* @return	BMSResult
*/
BMSResult BMSBrightness	(BMSHandle hDevice, short BrightF, short BrightR);

/** Changes acquisition contrast.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	ContrastF	Between -100 (lowest contrast) to +100.	Front side.
* @param	ContrastR	As ContrastF for Rear side.
* @return	BMSResult
*/
BMSResult BMSContrast	(BMSHandle hDevice, short ContrastF, short ContrastR);

/** Checks if a given OCR font/set is enabled.
* @return	BMSResult
*/
BMSResult BMSCheckOCRType(short OCRChar);

/** Chages current device settings. See K2S_PAR_ID. and corresponding ParVal values.
* @param	ParId			See K2S_PAR_ID.
* @param	ParVal			See values descibed in the corresponding K2S_PAR_ID. 
* @return	BMSResult
* @see K2S_PAR_ID
*/
BMSResult BMSSetParameter(unsigned short ParId, short ParVal);


/** Returns current device settings. See K2S_PAR_ID. and corresponding ParVal values.
* @param	ParId			See K2S_PAR_ID.
* @param	ParVal		(in out)	Current value. 
* @return	BMSResult
* @see K2S_PAR_ID
*/
BMSResult BMSGetParameter(unsigned short ParId, short *ParVal);

/** Acquisition and OCR recognition.<BR>
* <BR>
* Image requirements are to be set in advance:<BR>
* - color, using BMSSetParameter.<BR>
* - resolution, calling BMSImgCreate;<BR>
* <BR>
* OCR requirments are defined by the pDoc structure. See BMSOCRDoc.<BR>
* If no OCR is required, pass NULL as pDoc or set pDoc->nZn=0.
* <BR>
* @param	<b>hDevice</b>		BMSHandle returned by initialization function.
* @param	<b>pDoc</b>		(in out)	See BMSOCRDoc. NULL or pDoc->nZn = 0 to avoid OCR.<BR>
										On return, pDoc->Zn[i].CharRead reflects the number of chars read for the i-th zone.<BR>
										pDoc->Zn[i].OCRchar can change too (in case of multple fonts, like CMC7_OR_OCRB1_EB1, to reflect the font detected.
* @param	<b>cdl</b>		(in out)	Returns the codeline of pDoc->nZn, separated by a TAB.<BR>
										Not recognised chars are returned as '?' (and their confidence is set to 1).<BR>
										Buffer allocated by calling application (see size).
* @param	<b>cnf</b>		(in out)	As codeline or NULL. For each char, a confidence value is returned between 1 (not recognised) and 100.
* @param	<b>size</b>		(in out)	Size of Cdl and, if required, Cnf, including null terminator.
* @return	BMSResult
* @see BMSImgCreate
*/
BMSResult BMSImgScan1 (BMSHandle hDevice, BMSOCRDoc *pDoc, char *cdl, char *cnf, short size);


/** Sets Image requirements before calling BMSImgScan or BMSImgScan1<BR>
* It  allocates memory, to be released with BMSImgFree, after image is saved.
* @param	hDevice	BMSHandle returned by initialization function.
* @param	dpiF	Between 0 and HW resolution. If no front image is required, set to 0.
* @param	dpiR	As dpiF, but for rear image.
* @return	BMSResult
* @see BMSImgScan
* @see BMSImgFree
* @see BMSImgSave
* @see BMSImgSaveFR
* @see BMSImgGet
*/
BMSResult BMSImgCreate(BMSHandle hDevice, short dpiF, short dpiR);

/** Sets Image requirements before calling BMSImgScan or BMSImgScan1<BR>
* It  allocates memory, to be released with BMSImgFree, after image is saved.
* @param	hDevice	BMSHandle returned by initialization function.
* @param	dpiF	Between 0 and HW resolution. If no front image is required, set to 0.
* @param	dpiR	As dpiF, but for rear image.
* @param	dpiR	As dpiF, but for backlighted image.
* @return	BMSResult
* @see BMSImgScan
* @see BMSImgFree
* @see BMSImgSave
* @see BMSImgSaveFR
* @see BMSImgGet
*/
BMSResult BMSImgCreate1(BMSHandle hDevice, short dpiF, short dpiR, short dpiBL);

/** Releases resources allocated with BMSImgCreate.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	side		REAR or FRONT.
* @return	BMSResult
* @see BMSImgCreate
*/
BMSResult BMSImgFree	(BMSHandle hDevice);

/** Saves one side image of the document aquired with BMSImgScan.<BR><BR>
* @param <b>hDevice</b>	  BMSHandle returned by initialization function.
* @param <b>side</b>	  REAR or FRONT.
* @param <b>pRoi</b>	  Pointer to a Region of interest, if any. See BMSRoi. 
								Pass NULL if the whole image is required.
* @param <b>parameter</b> Depends of required format:<BR>
								JPEG, TIF_JPEG, PDF_JPEG -> quality. Between 1 and 100.<BR>
								BMP and TIFF formats -> image depth (1 - b&w, 8 - grayscale, 24 - color; 0 - default [8 if gray/red, 24 if color]).
* @param <b>format</b>	  value of K2S_IMG_FORMAT (required) | value of K2S_ROTATION (optional).
* @param <b>pFname</b>	  Complete filename, including path and extension. Path must exist.<BR><BR>
* @return	BMSResult
* @see BMSImgCreate
* @see BMSImgScan
* @see BMSImgSave
* @see BMSImgSaveFR
* @see BMSImgFree
* @see BMSImgGet
* @see BMSImgGetFree
*/
BMSResult BMSImgSave(BMSHandle hDevice, short side, BMSRoi* pRoi, 
							 short parameter, short format, const char* pFname);
BMSResult BMSImgSave2(BMSHandle hDevice, short side, BMSRoi Roi, 
							 short parameter, short format, const char* pFname);

/** Saves two side image of the document aquired with BMSImgScan, one above the other.<BR><BR>
* @param <b>hDevice</b>	  BMSHandle returned by initialization function.
* @param <b>sideUp</b>	  REAR or FRONT, side in the upper part of the final image.
* @param <b>sideDown</b>  REAR or FRONT, side in the lower part of the final image.
* @param <b>parameter</b> Depends of required format:<BR>
								JPEG, TIF_JPEG, PDF_JPEG -> quality. Between 1 and 100.<BR>
								BMP  and TIFF formats   -> image depth (1 - b&w, 8 - grayscale, 24 - color; 0 - default [8 if gray/red, 24 if color]).
* @param <b>format</b>	  value of K2S_IMG_FORMAT (required) | value of K2S_ROTATION (optional).
* @param <b>pFname</b>	  Complete filename, including path and extension. Path must exist.<BR><BR>
* @return	BMSResult
* @see BMSImgCreate
* @see BMSImgScan
* @see BMSImgFree
* @see BMSImgSave
* @see BMSImgSaveFR
* @see BMSImgGet
* @see BMSImgGetFree
*/
BMSResult BMSImgSaveFR	(BMSHandle hDevice, short sideUp, short sideDown,
								 short parameter, short format, const char* pFname);
/** Returns a memory buffer with one side image of the document aquired with BMSImgScan.<BR>
* After use, buffer must be realeased calling BMSImgGetFree.<BR><BR>
* @param <b>hDevice</b>	  BMSHandle returned by initialization function.
* @param <b>side</b>	  REAR or FRONT.
* @param <b>pRoi</b>	  Pointer to a Region of interest, if any. See BMSRoi. 
								Pass NULL if the whole image is required.
* @param <b>parameter</b> Depends of required format:<BR>
								JPEG and TIF_JPEG     -> quality. Between 1 and 100.<BR>
								BMP  and TIFF formats -> image depth (1 - b&w, 8 - grayscale, 24 - color; 0 - default [8 if gray/red, 24 if color]).
* @param <b>format</b>	  value of K2S_IMG_FORMAT (required) | value of K2S_ROTATION (optional).
* @param <b>nBytes</b>	  (in out)	Image buffer size.<BR><BR>
* @return	BMSResult
* @see BMSImgCreate
* @see BMSImgScan
* @see BMSImgFree
* @see BMSImgGet
* @see BMSImgGetFree
*/
unsigned char* BMSImgGet(BMSHandle hDevice, short side, BMSRoi* pRoi,
								short parameter, short format, int* nBytes);

/** Returns a memory buffer with one side image of the document aquired with BMSImgScan.<BR>
* After use, buffer must be realeased calling BMSImgGetFree.<BR><BR>
* @param <b>hDevice</b>	  BMSHandle returned by initialization function.
* @param <b>sideUp</b>	  REAR or FRONT, side in the upper part of the final image.
* @param <b>sideDown</b>  REAR or FRONT, side in the lower part of the final image.
								Pass NULL if the whole image is required.
* @param <b>parameter</b> Depends of required format:<BR>
								JPEG and TIF_JPEG     -> quality. Between 1 and 100.<BR>
								BMP  and TIFF formats -> image depth (1 - b&w, 8 - grayscale, 24 - color; 0 - default [8 if gray/red, 24 if color]).
* @param <b>format</b>	  value of K2S_IMG_FORMAT (required) | value of K2S_ROTATION (optional).
* @param <b>nBytes</b>	  (in out)	Image buffer size.<BR><BR>
* @return	BMSResult
* @see BMSImgCreate
* @see BMSImgScan
* @see BMSImgFree
* @see BMSImgGet
* @see BMSImgGetFree
*/
unsigned char* BMSImgGetFR(BMSHandle hDevice, short sideUp, short sideDown,
								short parameter, short format, int* nBytes);

/** Releases resources allocated with BMSImgGet
* @param	hDevice		BMSHandle returned by initialization function.
* @param	pImg		pointer returned by BMSImgGet.
* @return	BMSResult
* @see BMSImgCreate
* @see BMSImgScan
* @see BMSImgFree
* @see BMSImgGet
*/
BMSResult BMSImgGetFree(BMSHandle hDevice, unsigned char* pImg);

/** Changes image brightness, after scanning.
* @param	side	REAR or FRONT.
* @param	val		Between -100 (darkest) to +100.
* @return	BMSResult
*/
BMSResult BMSImgBright (short side, short val);

/** Changes image contrast, after scanning.
* @param	side	REAR or FRONT.
* @param	val		Between -100 (lowest contrast) to +100.
* @return	BMSResult
*/
BMSResult BMSImgContr  (short side, short val);

/** Moves a document forward or backward in the K2s.
* @param	hDevice				BMSHandle returned by initialization function.	
* @param	forward				>0 forward, <=0 backward.
* @param	stopWithPhotocell			>0 with, <=0 ignore.
* @param	time				in milleseconds.		
* @return	BMSResult
*/
BMSResult BMSMoveScanner_mSec	(BMSHandle hDevice, short forward, short stopWithPhotocell, int time);

#ifndef K2S_MGSTR_SENTINEL
#define K2S_MGSTR_SENTINEL
/** BMSMagStripe flags. 			*/
enum K2S_MGSTR_FLAG {
	/** Removes start sentinel char.	*/
	K2S_MGSTR_NO_START	= 0x0001,
	/** Removes stop sentinel char.		*/
	K2S_MGSTR_NO_STOP	= 0x0002,
	/**	Removes LCR char		*/
	K2S_MGSTR_NO_LCR	= 0x0004,
};
#endif // K2S_MGSTR_SENTINEL

/** Cleaning functionality.<BR>
* It performs forward and backward movements, and requires the distributed cleaning cloth. 
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	BMSResult
*/
BMSResult BMSClean(BMSHandle hDevice);

/**  Check if K2s need to be cleaned. Implemented for  CMC7 recognition.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	BMSResult	(out)	BMS_WAR_TOCLEAN if dirty, BMS__OK if not dirty, BMS_ERR in other cases (like device not opened).
*/
BMSResult BMSCheckCleaning	(BMSHandle hDevice);

/** Get current configuration directory (i.e. containing databases and calibration files).
* @param	hDevice			BMSHandle returned by initialization function.	
* @param	ConfigDir	(in out)	configuration directory.	
* @return	BMSResult
*/
BMSResult BMSConfigDirGet	(BMSHandle hDevice, char ConfigDir[MAX_PATH]);
/** Returns the total number of documents acquired on this PC with this device. 
* The number is stopped at 0x7FFFFFFF = 2147483647.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	int
*/

/**  Generate a log depicting current installation data, including dll and databases information.
* @param	fname		(in)	Complete filename, including path and extension. Path must exist.<BR><BR>
* @return	BMSResult
*/
BMSResult BMSGenerateInstLog(char fname[MAX_PATH]);

int BMSPagNumTot (BMSHandle hDevice);

/** Detects blank pages. 
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	side		FRONT or REAR (otherwise return code is BMS_ERR_INV_PARAMETER)	
* @param	PixThresholdbetween 0 and 100 (otherwise return code is BMS_ERR_INV_PARAMETER). 
								If 0, almost no blank page will be detected. If 100, most pages will be detected as blank.
* @param	GrayThresholdbetween 0 (black) and 255 (white)  (otherwise return code is BMS_ERR_INV_PARAMETER).
* @return	BMS_ERR_INV_PARAMETER, BMS_ERR, BMS__OK (page is not blank), BMS_WAR_BLANK_PAGE (page seems blank)
*/
BMSResult BMSIsPageBlank(BMSHandle hDevice, short side, unsigned char PixThreshold, unsigned char GrayThreshold);


/** Moves of about nLines at a given resolution (dpi), forward or backward.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	forward		> 0 means forward, <= 0 backward	
* @param	nLines		n. of lines at the dpi resolution. It is rounded by the SW.
* @param	dpi			resolution of the desired nLines
* @return	BMS_ERR_INV_PARAMETER, BMS_ERR, BMS__OK
*/
BMSResult BMSMoveScannerLines(BMSHandle hDevice, short forward, short nLines, short dpi);


#define GENERIC_TRACK_SIZE	128
/** Returns data acquired with the magnetic stripe reader.
* If no card is passed withint given timeout (see K2S_PAR_TIMEOUT_MAGSTRIPE), it returns with error.<BR>
* To know if a given track reader is present, use BMSReadStatusL - K2sStatus - K2S_BDG_T1ACTIVE and similar.
* To get in advance information about presence of data, use BMSReadStatusL - K2S_BDG_T1READY and similar.
* @param	hDevice			BMSHandle returned by initialization function.	
* @param	Trace1		(in out)	Data on track 1. If NULL, skipped. If not be NULL, GENERIC_TRACK_SIZE is suggested as size. Error if too short.
* @param	Trace2		(in out)	Data on track 2. If NULL, skipped. If not be NULL, GENERIC_TRACK_SIZE is suggested as size. Error if too short.
* @param	Trace3		(in out)	Data on track 3. If NULL, skipped. If not be NULL, GENERIC_TRACK_SIZE is suggested as size. Error if too short.
* @param	flags			See K2S_MGSTR_FLAG
* @return	BMSResult
* @see BMSReadStatusL
* @see K2sStatus
* @see K2S_MGSTR_FLAG
*/
BMSResult BMSMagStripeC(BMSHandle hDevice, char *Trace1, char *Trace2, char *Trace3, unsigned char flags, unsigned short Size);

/** Removes read data, if any, without reading.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	BMSResult
*/
BMSResult BMSMagStripeReset(BMSHandle hDevice);

/**  Use this function to save multipage TIF files, befor adding pages. 
* Then call BMSMultiPageClose. Added in ver. 2.1.1.2.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	fName		Complete filename of the final image, including path and extension. Path must exist.
* @param	encoding	TIF or TIF_JPEG or TIF_CCITTFAX3 or TIF_CCITTFAX4 or PDF_JPEG.
* @param	param		In case of TIF_JPEG or PDF_JPEG, quality (valid values between 1 and 100). Not used for other formats.
* @return	BMSResult
* @see		BMSMultiPageAdd
* @see		BMSMultiPageClose
*/
BMSResult BMSMultiPageInit0	(BMSHandle hDevice, const char* fName, short encoding, short param);

/**  Use this function to add pages to an image file created with BMSMultiPageInit.
* Then call BMSMultiPageClose. Added in ver. 2.1.1.2.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	fNamePage	Complete filename of the page to be added, including path and extension. File must exist.
* @return	BMSResult
* @see		BMSMultiPageInit
* @see		BMSMultiPageClose
*/
BMSResult BMSMultiPageAdd	(BMSHandle hDevice, const char* fNamePage);

/**  Closes the multipage image file created with BMSMultiPageInit. 
* Added in ver. 2.1.1.2.
* @param	hDevice		BMSHandle returned by initialization function.	
* @return	BMSResult
* @see		BMSMultiPageInit
* @see		BMSMultiPageAdd
*/
BMSResult BMSMultiPageClose	(BMSHandle hDevice);

/** Detects blank pages. 
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	side		FRONT or REAR (otherwise return code is BMS_ERR_INV_PARAMETER)	
* @param	PixThresholdbetween 0 and 100 (otherwise return code is BMS_ERR_INV_PARAMETER). 
								If 0, almost no blank page will be detected. If 100, most pages will be detected as blank.
* @param	GrayThresholdbetween 0 (black) and 255 (white)  (otherwise return code is BMS_ERR_INV_PARAMETER).
* @return	BMS_ERR_INV_PARAMETER, BMS_ERR, BMS__OK (page is not blank), BMS_WAR_BLANK_PAGE (page seems blank)
*/
BMSResult BMSIsPageBlank(BMSHandle hDevice, short side, unsigned char  PixThreshold, unsigned char  GrayThreshold);

/**  Inketjet1 setup. To be called before document scanning.
* BMSInkjet1SetUp2 added in ver. 2.1.5.8.
* BMSInkjet1SetUp1 added in ver. 2.1.4.0.
* See DEVICE_INKJET1 and BMSCompositionL to detect if the option is present.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	stamp		string to be printed.	
* @param	font		font name, i.e "arial".	
* @param	size		font size.	Printer has 12 nooozels. Larger font will be cut.
* @param	strech		0 normal size; < 0 condensed; > 0 expanded
* @param	offset		delay in start printing.
* @param	addtoimage	>0 if string has to be added to document image too.
* @param	nFire		0 standard fire; 1 or 2 to get heavier drop
* @param	mirror		>0 if string has to be mirrored
* @return	BMSResult
*/
BMSResult BMSInkjet1SetUp2 (BMSHandle hDevice, const char* stamp, char *font, short size, short stretch, short offset, short addtoimage, short nFire, short mirror);

/**  Inketjet1 setup, using an image file. To be called before document scanning.
* Added in ver. 2.1.6.0.
* See DEVICE_INKJET1 and BMSCompositionL to detect if the option is present.
* @param	hDevice		BMSHandle returned by initialization function.
* @param	fname		Complete filename, including path and extension.
* @param	size		font size.	Printer has 12 nooozels. Larger font will be cut.
* @param	strech		0 normal size; < 0 condensed; > 0 expanded
* @param	offset		delay in start printing.
* @param	addtoimage	>0 if string has to be added to document image too.
* @param	nFire		0 standard fire; 1 or 2 to get heavier drop
* @param	mirror		>0 if string has to be mirrored
* @return	BMSResult
*/
BMSResult BMSInkjet1SetUpImg (BMSHandle hDevice, const char* fname, short stretch, short offset, short addtoimage, short nFire, short mirror);

/**  Inketjet2 setup. To be called before document scanning.
* Added in ver. 2.1.5.5. See DEVICE_INKJET2 and BMSCompositionL to detect if the option is present.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	fname		(in out)	Complete filename, including path and extension.
* @param	offset		delay in start printing.
* @param	addtoimage	rfu; not implemented at present; >0 if string has to be added to document image too.
* @return	BMSResult
*/
BMSResult BMSInkjet2SetUpImg	(BMSHandle hDevice, const char*fname, short offset, short addtoimage);

/**  Inketjet2 setup. To be called before document scanning. To be implemented.
* Added in ver. 2.1.5.5. See DEVICE_INKJET2 and BMSCompositionL to detect if the option is present.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	stamp		string to be printed.	
* @param	font		font name, i.e "arial".	
* @param	size		font size.
* @param	offset		delay in start printing.
* @param	addtoimage	>0 if string has to be added to document image too.
* @return	BMSResult
*/
BMSResult BMSInkjet2SetUp	(BMSHandle hDevice, const char* stamp, 
				 char *font, short size, short offset, short addtoimage);

/** Get current OCR databases directory. Default is Config directory.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	DBDir		if NULL, it just returns the directory len for memory allocation.
* @param	size		if <= 0, it just returns the directory len for memory allocation.
* @return	lenght of DBDir, -1 in case of error (for example if DBDir != NULL and size is less then required lenght).
*/
short BMSDBDirGet(BMSHandle hDevice, char*DBDir, short size);

/** Get current OCR databases directory. Default is Config directory.
* @param	hDevice		BMSHandle returned by initialization function.	
* @param	DBDir		New OCR databases directory. It must be shorter then MAX_PATH.
* @return	BMS_ERR_INV_PARAMETER, BMS__OK
*/
BMSResult BMSDBDirSet(BMSHandle hDevice, const char*DBDir);


BMSResult BMSLogFileName (char LogFName[MAX_PATH]);
#if defined (__cplusplus)
}
#endif

#endif // K2SLIB
